// Protocol Buffers - Google's data interchange format
// Copyright 2008 Google Inc.  All rights reserved.
// https://developers.google.com/protocol-buffers/
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#ifndef GOOGLE_PROTOBUF_STUBS_STATUS_H_
#define GOOGLE_PROTOBUF_STUBS_STATUS_H_

#include <string>

#include <herogoogle/protobuf/stubs/stringpiece.h>



namespace herogoogle {
namespace protobuf {
namespace util {
namespace status_internal {

// These values must match error codes defined in google/rpc/code.proto.
enum class StatusCode : int {
  kOk = 0,
  kCancelled = 1,
  kUnknown = 2,
  kInvalidArgument = 3,
  kDeadlineExceeded = 4,
  kNotFound = 5,
  kAlreadyExists = 6,
  kPermissionDenied = 7,
  kUnauthenticated = 16,
  kResourceExhausted = 8,
  kFailedPrecondition = 9,
  kAborted = 10,
  kOutOfRange = 11,
  kUnimplemented = 12,
  kInternal = 13,
  kUnavailable = 14,
  kDataLoss = 15,
};

class CSDKPROTOBUF_API Status {
 public:
  // Creates a "successful" status.
  Status();

  // Create a status in the canonical error space with the specified
  // code, and error message.  If "code == 0", error_message is
  // ignored and a Status object identical to Status::kOk is
  // constructed.
  Status(StatusCode error_code, StringPiece error_message);
  Status(const Status&);
  Status& operator=(const Status& x);
  ~Status() {}

  // Accessor
  bool ok() const { return error_code_ == StatusCode::kOk; }
  StatusCode code() const { return error_code_; }
  StringPiece message() const {
    return error_message_;
  }

  bool operator==(const Status& x) const;
  bool operator!=(const Status& x) const {
    return !operator==(x);
  }

  // Return a combination of the error code name and message.
  std::string ToString() const;

 private:
  StatusCode error_code_;
  std::string error_message_;
};

// Returns an OK status, equivalent to a default constructed instance. Prefer
// usage of `OkStatus()` when constructing such an OK status.
CSDKPROTOBUF_API Status OkStatus();

// Prints a human-readable representation of 'x' to 'os'.
CSDKPROTOBUF_API std::ostream& operator<<(std::ostream& os, const Status& x);

// These convenience functions return `true` if a given status matches the
// `StatusCode` error code of its associated function.
CSDKPROTOBUF_API bool IsAborted(const Status& status);
CSDKPROTOBUF_API bool IsAlreadyExists(const Status& status);
CSDKPROTOBUF_API bool IsCancelled(const Status& status);
CSDKPROTOBUF_API bool IsDataLoss(const Status& status);
CSDKPROTOBUF_API bool IsDeadlineExceeded(const Status& status);
CSDKPROTOBUF_API bool IsFailedPrecondition(const Status& status);
CSDKPROTOBUF_API bool IsInternal(const Status& status);
CSDKPROTOBUF_API bool IsInvalidArgument(const Status& status);
CSDKPROTOBUF_API bool IsNotFound(const Status& status);
CSDKPROTOBUF_API bool IsOutOfRange(const Status& status);
CSDKPROTOBUF_API bool IsPermissionDenied(const Status& status);
CSDKPROTOBUF_API bool IsResourceExhausted(const Status& status);
CSDKPROTOBUF_API bool IsUnauthenticated(const Status& status);
CSDKPROTOBUF_API bool IsUnavailable(const Status& status);
CSDKPROTOBUF_API bool IsUnimplemented(const Status& status);
CSDKPROTOBUF_API bool IsUnknown(const Status& status);

// These convenience functions create an `Status` object with an error code as
// indicated by the associated function name, using the error message passed in
// `message`.
//
// These functions are intentionally named `*Error` rather than `*Status` to
// match the names from Abseil:
// https://github.com/abseil/abseil-cpp/blob/2e9532cc6c701a8323d0cffb468999ab804095ab/absl/status/status.h#L716
CSDKPROTOBUF_API Status AbortedError(StringPiece message);
CSDKPROTOBUF_API Status AlreadyExistsError(StringPiece message);
CSDKPROTOBUF_API Status CancelledError(StringPiece message);
CSDKPROTOBUF_API Status DataLossError(StringPiece message);
CSDKPROTOBUF_API Status DeadlineExceededError(StringPiece message);
CSDKPROTOBUF_API Status FailedPreconditionError(StringPiece message);
CSDKPROTOBUF_API Status InternalError(StringPiece message);
CSDKPROTOBUF_API Status InvalidArgumentError(StringPiece message);
CSDKPROTOBUF_API Status NotFoundError(StringPiece message);
CSDKPROTOBUF_API Status OutOfRangeError(StringPiece message);
CSDKPROTOBUF_API Status PermissionDeniedError(StringPiece message);
CSDKPROTOBUF_API Status ResourceExhaustedError(StringPiece message);
CSDKPROTOBUF_API Status UnauthenticatedError(StringPiece message);
CSDKPROTOBUF_API Status UnavailableError(StringPiece message);
CSDKPROTOBUF_API Status UnimplementedError(StringPiece message);
CSDKPROTOBUF_API Status UnknownError(StringPiece message);

}  // namespace status_internal

using ::herogoogle::protobuf::util::status_internal::Status;
using ::herogoogle::protobuf::util::status_internal::StatusCode;

using ::herogoogle::protobuf::util::status_internal::IsAborted;
using ::herogoogle::protobuf::util::status_internal::IsAlreadyExists;
using ::herogoogle::protobuf::util::status_internal::IsCancelled;
using ::herogoogle::protobuf::util::status_internal::IsDataLoss;
using ::herogoogle::protobuf::util::status_internal::IsDeadlineExceeded;
using ::herogoogle::protobuf::util::status_internal::IsFailedPrecondition;
using ::herogoogle::protobuf::util::status_internal::IsInternal;
using ::herogoogle::protobuf::util::status_internal::IsInvalidArgument;
using ::herogoogle::protobuf::util::status_internal::IsNotFound;
using ::herogoogle::protobuf::util::status_internal::IsOutOfRange;
using ::herogoogle::protobuf::util::status_internal::IsPermissionDenied;
using ::herogoogle::protobuf::util::status_internal::IsResourceExhausted;
using ::herogoogle::protobuf::util::status_internal::IsUnauthenticated;
using ::herogoogle::protobuf::util::status_internal::IsUnavailable;
using ::herogoogle::protobuf::util::status_internal::IsUnimplemented;
using ::herogoogle::protobuf::util::status_internal::IsUnknown;

using ::herogoogle::protobuf::util::status_internal::AbortedError;
using ::herogoogle::protobuf::util::status_internal::AlreadyExistsError;
using ::herogoogle::protobuf::util::status_internal::CancelledError;
using ::herogoogle::protobuf::util::status_internal::DataLossError;
using ::herogoogle::protobuf::util::status_internal::DeadlineExceededError;
using ::herogoogle::protobuf::util::status_internal::FailedPreconditionError;
using ::herogoogle::protobuf::util::status_internal::InternalError;
using ::herogoogle::protobuf::util::status_internal::InvalidArgumentError;
using ::herogoogle::protobuf::util::status_internal::NotFoundError;
using ::herogoogle::protobuf::util::status_internal::OkStatus;
using ::herogoogle::protobuf::util::status_internal::OutOfRangeError;
using ::herogoogle::protobuf::util::status_internal::PermissionDeniedError;
using ::herogoogle::protobuf::util::status_internal::ResourceExhaustedError;
using ::herogoogle::protobuf::util::status_internal::UnauthenticatedError;
using ::herogoogle::protobuf::util::status_internal::UnavailableError;
using ::herogoogle::protobuf::util::status_internal::UnimplementedError;
using ::herogoogle::protobuf::util::status_internal::UnknownError;

}  // namespace util
}  // namespace protobuf
}  // namespace google



#endif  // GOOGLE_PROTOBUF_STUBS_STATUS_H_
