#include "HeroUSDKCommandlet.h"

#include <string>
#include "HttpManager.h"
#include "HeroUsdkSupportKit.h"
#include "HttpModule.h"
#include "XmlFile.h"
#include "Data/ErrorCode.h"
#include "Dom/JsonObject.h"
#include "Dom/JsonValue.h"
#include "Interfaces/IHttpResponse.h"
#include "Interfaces/IPluginManager.h"
#include "Misc/FileHelper.h"
#include "Misc/Paths.h"
#include "Serialization/JsonWriter.h"
#include "Serialization/JsonSerializer.h"
#include "Setting/UsdkSettings.h"

DEFINE_LOG_CATEGORY_STATIC(HeroUSDKCommandlet, All, All);
constexpr float HttpRequestTimeout = 60.f;
constexpr float HttpRequestTickDelta = 1.f;

UHeroUSDKCommandlet::UHeroUSDKCommandlet(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer)
{
	IsClient = false;
	IsServer = false;
}

static void PrintUsage()
{
	UE_LOG(HeroUSDKCommandlet, Display, TEXT("Linter Usage: {Editor}.exe Project.uproject -run=Linter \"/Game/\""));
	UE_LOG(HeroUSDKCommandlet, Display, TEXT(""));
	UE_LOG(HeroUSDKCommandlet, Display, TEXT("This will run the Linter on the provided project and will scan the supplied directory, example being the project's full Content/Game tree. Can add multiple paths as additional arguments."));
}

static void GenAndroidCfgFile(const FString& Config)
{
	TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject);
	const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Config);
	if (!FJsonSerializer::Deserialize(Reader, JsonObject))
	{
		UE_LOG(LogTemp, Error, TEXT("设置安卓配置文件失败，json字符串格式错误,Json:%s"), *Config);
		return;
	}
	
	const FString ProductId = JsonObject->GetStringField(TEXT("productId"));
	const FString ProductKey = JsonObject->GetStringField(TEXT("productKey"));

	if (ProductId.IsEmpty() || ProductKey.IsEmpty())
	{
		UE_LOG(LogTemp, Error, TEXT("设置安卓配置文件失败，ProductId：%s, ProductKey：%s"), *ProductId, *ProductKey);
		return;
	}

	const FString BaseDir = IPluginManager::Get().FindPlugin("HeroUSDKPlugin")->GetBaseDir();
	const FString FilePath = FPaths::ConvertRelativePathToFull(FPaths::Combine(*BaseDir, TEXT("Source/HeroUSDKPlugin/Android/assets/herosdkcfg.xml")));
	if (!FPaths::FileExists(FilePath))
	{
		UE_LOG(LogTemp, Error, TEXT("herosdkcfg.xml 文件不存在,路径：%s"), *FilePath);
		return;
	}

	const TSharedPtr<FXmlFile> File = MakeShareable(new FXmlFile(FilePath));
	if (!File)
	{
		UE_LOG(LogTemp, Error, TEXT("herosdkcfg.xml 文件格式有问题,路径：%s"), *FilePath);
		return;
	}

	const FXmlNode* RootNode = File->GetRootNode();
	TArray<FXmlNode*> Nodes = RootNode->GetChildrenNodes();
	bool bIsEditXml = false;
	for (int32 i = 0; i < Nodes.Num(); i++)
	{
		FXmlNode* NodeItem = Nodes[i];
		FString Key = NodeItem->GetAttribute("name");
		if (Key.Equals("_hu_pid_"))
		{
			NodeItem->SetContent(ProductId);
			bIsEditXml = true;
			UE_LOG(LogTemp, Log, TEXT("------> _hu_pid_ : %s <--------"), *ProductId);
		}
		if (Key.Equals("_hu_pk_"))
		{
			NodeItem->SetContent(ProductKey);
			bIsEditXml = true;
			UE_LOG(LogTemp, Log, TEXT("------> _hu_pk_ : %s <--------"), *ProductKey);
		}
	}
	if (bIsEditXml)
	{
		File->Save(FilePath);
		UE_LOG(LogTemp, Log, TEXT("------> Update herosdkcfg Xml  <--------"));
	}
}

int32 UHeroUSDKCommandlet::Main(const FString& InParams)
{
	FString Params = InParams;
	// Parse command line.
	TArray<FString> Paths;
	TArray<FString> Switches;
	TMap<FString, FString> ParamsMap;
	UCommandlet::ParseCommandLine(*Params, Paths, Switches, ParamsMap);
	UE_LOG(HeroUSDKCommandlet, Display, TEXT("HeroUSDKCommandlet is indeed running!"));
	FString OneKeyUrl;
	FString BuildChannelStr;
	if (!FParse::Value(*Params, TEXT("OneKey="), OneKeyUrl))
	{
		UE_LOG(HeroUSDKCommandlet, Display, TEXT("Run HeroUSDKCommandlet with no OneKey param, get OneKey from Game.ini"));
		OneKeyUrl = GetMutableDefault<UUsdkSettings>()->PlatformAccessUrl;
	}
	if (FParse::Value(*Params, TEXT("Channel="), BuildChannelStr))
	{
		UE_LOG(HeroUSDKCommandlet, Display, TEXT("Run HeroUSDKCommandlet with Channel %s"), *BuildChannelStr);
	
		EHeroUSDKChannel Channel = {};
		if (BuildChannelStr == TEXT("Mainland"))
		{
			Channel = EHeroUSDKChannel::China;
		}
		else if (BuildChannelStr == TEXT("Overseas"))
		{
			Channel = EHeroUSDKChannel::Global;
		}
		else if (BuildChannelStr == TEXT("Steam"))
		{
			Channel = EHeroUSDKChannel::Steam;
		}
		else if (BuildChannelStr == TEXT("BiliBili"))
		{
			Channel = EHeroUSDKChannel::Bilibili;
		}
		else
		{
			UE_LOG(LogTemp, Warning, TEXT("Invalid BUILD_CHANNEL value, defaulting to None"));
		}

		GetMutableDefault<UUsdkSettings>()->Channel = Channel;
	}
	
	if (OneKeyUrl.IsEmpty())
	{
		PrintUsage();
		UE_LOG(HeroUSDKCommandlet, Error, TEXT("Hero Usdk 请配置中台参数 "));
		return -1;
	}
	FString Url;
	//base64解码获取请求地址
	FBase64::Decode(OneKeyUrl, Url);
	UE_LOG(HeroUSDKCommandlet, Display, TEXT(" Base64 Decode : %s"), *Url);
	if (!Url.IsEmpty())
	{
		FHttpModule* Http = &FHttpModule::Get();
		TSharedRef<IHttpRequest, ESPMode::ThreadSafe> HttpRequest = Http->CreateRequest();
		HttpRequest->SetURL(Url);
		HttpRequest->SetVerb("GET");
		HttpRequest->SetTimeout(HttpRequestTimeout);
		HttpRequest->OnProcessRequestComplete().BindUObject(this, &UHeroUSDKCommandlet::OnRequestComplete);
		HttpRequest->ProcessRequest();
		float Time = 15.f;
		while (Time > 0)
		{
			Http->GetHttpManager().Tick(HttpRequestTickDelta);
			FPlatformProcess::Sleep(HttpRequestTickDelta);
			Time--;
		}
	}

	return 0;
}

void UHeroUSDKCommandlet::OnRequestComplete(FHttpRequestPtr Request, FHttpResponsePtr Response, bool bSuccess)
{
	if (!bSuccess)
	{
		UE_LOG(LogTemp, Error, TEXT("获取USDK配置请求失败 "));
		return;
	}

	const FString ContentStr = Response->GetContentAsString();
	UE_LOG(LogTemp, Display, TEXT("中台参数Response : %s"), *ContentStr);
	
	TSharedPtr<FJsonObject> ResultObject = MakeShareable(new FJsonObject);
	const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ContentStr);
	if (FJsonSerializer::Deserialize(Reader, ResultObject))
	{
		EUsdkErrorCode Code = EUsdkErrorCode::Failed;
		Code = static_cast<EUsdkErrorCode>(ResultObject->GetIntegerField(TEXT("code")));
		if (Code == EUsdkErrorCode::Success)
		{
			const FString Data = ResultObject->GetStringField(TEXT("data"));
			
			TArray<uint8> content;
			FBase64::Decode(Data, content);
			const std::string cstr(reinterpret_cast<const char*>(content.GetData()), content.Num());  
			FString Result = UTF8_TO_TCHAR(cstr.c_str());
#if UE_EDITOR
			HeroUsdkSupportKit::GetInstance().GenInitConfig(Result);
#endif
			GetMutableDefault<UUsdkSettings>()->UsdkConfig = Result;
			GetMutableDefault<UUsdkSettings>()->SaveConfig(CPF_Config,*FPaths::Combine(FPaths::ProjectConfigDir(),TEXT("DefaultGame.ini")));
			UE_LOG(LogTemp, Display, TEXT(" 中台参数 Data : %s"), *Result);
			
			GenAndroidCfgFile(Result);
			return;
		}
		
		UE_LOG(LogTemp, Error, TEXT("获取USDK配置失败,Code:%d "),Code);
	}
}