// Fill out your copyright notice in the Description page of Project Settings.

#include "HeroUSDKCustomization.h"

#include <string>

#include "HeroUsdkSupportKit.h"
#include "HttpModule.h"
#include "Data/ErrorCode.h"
#include "Interfaces/IHttpRequest.h"
#include "Interfaces/IHttpResponse.h"
#include "Interfaces/IPluginManager.h"
#include "Setting/UsdkSettings.h"

#if WITH_EDITOR

#include "Widgets/DeclarativeSyntaxSupport.h"
#include "Widgets/SBoxPanel.h"
#include "Widgets/Layout/SWrapBox.h"
#include "Widgets/Input/SButton.h"
#include "DetailLayoutBuilder.h"
#include "DetailWidgetRow.h"
#include "DetailCategoryBuilder.h"
#include "Widgets/Input/SComboBox.h"
#define LOCTEXT_NAMESPACE "PlatformConfigEditor"



#define CONFIG_PATH GGameIni

FReply FHeroUSDKCustomization::RequestConfig()
{
	if (GetMutableDefault<UUsdkSettings>()->PlatformAccessUrl.IsEmpty())
	{
		UE_LOG(LogTemp, Error, TEXT("Hero Usdk 请配置中台参数 "));
		return FReply::Handled();
	}
	FString Url;
	//base64解码获取请求地址
	FBase64::Decode(GetMutableDefault<UUsdkSettings>()->PlatformAccessUrl, Url);
	UE_LOG(LogTemp, Log, TEXT(" Base64 Decode : %s"), *Url);
	if (!Url.IsEmpty())
	{
		const auto HttpRequest = FHttpModule::Get().CreateRequest();
		HttpRequest->SetURL(Url);
		HttpRequest->SetVerb("GET");
		HttpRequest->OnProcessRequestComplete().BindStatic(&OnRequestComplete);
		HttpRequest->ProcessRequest();
	}
	return FReply::Handled();
}

void FHeroUSDKCustomization::OnRequestComplete(FHttpRequestPtr Request, FHttpResponsePtr Response, bool bSuccess)
{
	if (!bSuccess)
	{
		UE_LOG(LogTemp, Error, TEXT("获取USDK配置请求失败 "));
		return;
	}

	const FString ContentStr = Response->GetContentAsString();
	UE_LOG(LogTemp, Log, TEXT("中台参数Response : %s"), *ContentStr);


	
	TSharedPtr<FJsonObject> ResultObject = MakeShareable(new FJsonObject);
	const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ContentStr);
	if (FJsonSerializer::Deserialize(Reader, ResultObject))
	{
		EUsdkErrorCode Code = EUsdkErrorCode::Failed;
		Code = static_cast<EUsdkErrorCode>(ResultObject->GetIntegerField(TEXT("code")));
		if (Code == EUsdkErrorCode::Success)
		{
			const FString Data = ResultObject->GetStringField(TEXT("data"));
			
			TArray<uint8> content;
			FBase64::Decode(Data, content);
			const std::string cstr(reinterpret_cast<const char*>(content.GetData()), content.Num());  
			FString Result = UTF8_TO_TCHAR(cstr.c_str());

			HeroUsdkSupportKit::GetInstance().GenInitConfig(Result);
			GetMutableDefault<UUsdkSettings>()->UsdkConfig = Result;
			GetMutableDefault<UUsdkSettings>()->SaveConfig(CPF_Config,*FPaths::Combine(FPaths::ProjectConfigDir(),TEXT("DefaultGame.ini")));
			UE_LOG(LogTemp, Log, TEXT(" 中台参数 Data : %s"), *Result);
			
			GenAndroidCfgFile(Result);

			TSharedPtr<FJsonObject> JsonObjectNew = MakeShareable(new FJsonObject);
			const TSharedRef<TJsonReader<>> Readernew = TJsonReaderFactory<>::Create(Result);
			if (!FJsonSerializer::Deserialize(Readernew, JsonObjectNew))
			{
				UE_LOG(LogTemp, Error, TEXT("设置安卓配置文件失败，json字符串格式错误,Json:%s"), *Result);
				return;
			}
			auto iheroglobal = JsonObjectNew->GetObjectField(GetMutableDefault<UUsdkSettings>()->Channel==EHeroUSDKChannel::Global?TEXT("ihero"):TEXT("iheroglobal"));
			if (iheroglobal->TryGetField(TEXT("asset.ui.logo2")))
			{
				FString iconpath=iheroglobal->GetStringField(TEXT("asset.ui.logo2"));
			
				GetMutableDefault<UUsdkSettings>()->iconPath = iconpath;
				GetMutableDefault<UUsdkSettings>()->SaveConfig(CPF_Config,*FPaths::Combine(FPaths::ProjectConfigDir(),TEXT("DefaultGame.ini")));
			}
			return;
			
		}
		
		UE_LOG(LogTemp, Error, TEXT("获取USDK配置失败,Code:%d "),Code);
	}
}

void FHeroUSDKCustomization::CustomizeDetails(const TSharedPtr<IDetailLayoutBuilder>& DetailBuilder)
{
	IDetailCustomization::CustomizeDetails(DetailBuilder);
	weakLayout = DetailBuilder;
}

TSharedRef<SWidget> FHeroUSDKCustomization::OnGenerateComboWidget(TSharedPtr<FString> InComboString)
{
	return
		SNew(SBox)
		[
			SNew(STextBlock)
			.Text(FText::FromString(*InComboString))
		];
}

void FHeroUSDKCustomization::OnSelectionChanged(TSharedPtr<FString> InSelectedItem, ESelectInfo::Type SelectInfo)
{
	auto setting = GetMutableDefault<UUsdkSettings>();
	if (InSelectedItem)
	{
		

		if (*InSelectedItem == TEXT("不选择镜像渠道"))
		{
			setting->CurrMirror.Empty();
			setting->MirrorConfig.Empty();
		}
		else
		{
			setting->CurrMirror = *InSelectedItem;
			TSharedPtr<FJsonObject> ResultObject = MakeShareable(new FJsonObject);
			const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(setting->UsdkConfig);
			if (FJsonSerializer::Deserialize(Reader, ResultObject))
			{
				if ( setting->Channel == EHeroUSDKChannel::China)
				{
					auto mirror = ResultObject->GetArrayField("heropc_img");
					for (auto& item:mirror)
					{
						auto itemObj = item->AsObject();
						if (itemObj->GetStringField( "name") == setting->CurrMirror)
						{
							ResultObject->SetObjectField("herogpc",itemObj);
							break;
						}
					}

					TSharedRef<TJsonWriter<TCHAR,TCondensedJsonPrintPolicy<TCHAR>>> jsonWriter = TJsonWriterFactory<TCHAR,TCondensedJsonPrintPolicy<TCHAR>>::Create(&setting->MirrorConfig);
					FJsonSerializer::Serialize(ResultObject.ToSharedRef(), jsonWriter);
				}
				else if (setting->Channel == EHeroUSDKChannel::Global)
				{
					auto mirror = ResultObject->GetArrayField("heroglobalpc_img");
					for (auto& item:mirror)
					{
						auto itemObj = item->AsObject();
						if (itemObj->GetStringField( "name") == setting->CurrMirror)
						{
							ResultObject->SetObjectField("heroglobalpc",itemObj);
							break;
						}
					}

					TSharedRef<TJsonWriter<TCHAR,TCondensedJsonPrintPolicy<TCHAR>>> jsonWriter = TJsonWriterFactory<TCHAR,TCondensedJsonPrintPolicy<TCHAR>>::Create(&setting->MirrorConfig);
					FJsonSerializer::Serialize(ResultObject.ToSharedRef(), jsonWriter);
				}

			}
			
		}
		setting->SaveConfig(CPF_Config,*FPaths::Combine(FPaths::ProjectConfigDir(),TEXT("DefaultGame.ini")));
	}
	

	
}

TSharedPtr<FString> FHeroUSDKCustomization::GetCurrentlySelectedItem()
{
	auto setting = GetMutableDefault<UUsdkSettings>();
	if (setting->CurrMirror.Len())
	{
		return MakeShared<FString>(GetMutableDefault<UUsdkSettings>()->CurrMirror);
	}

	return nullptr;
}

FText FHeroUSDKCustomization::GetComboBoxSelectedOptionText() const
{
	TSharedPtr<FString> SelectedItem = ComboBox->GetSelectedItem();
	if (SelectedItem.IsValid())
	{
		return FText::FromString(*SelectedItem);
	}

	if (MirrorChannels.Num()>0)
	{
		return FText::FromString(TEXT("请选择镜像渠道"));
	}
	return FText::FromString(TEXT("没有镜像渠道"));
}


TSharedRef<IDetailCustomization> FHeroUSDKCustomization::MakeInstance()
{
	
	static TSharedRef<IDetailCustomization> Singleton = MakeShareable(new FHeroUSDKCustomization());
	return Singleton;
}


void FHeroUSDKCustomization::ParseMirror()
{
	TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject);
	const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(GetMutableDefault<UUsdkSettings>()->UsdkConfig);
	if (!FJsonSerializer::Deserialize(Reader, JsonObject))
	{
		UE_LOG(LogTemp, Error, TEXT("UsdkConfig json格式错误"));
		return;
	}
	MirrorChannels.Empty();
	if (ComboBox)
	{
		ComboBox->ClearSelection();
	}

	auto setting = GetMutableDefault<UUsdkSettings>();
	if ( setting->Channel == EHeroUSDKChannel::China)
	{
		auto mirror = JsonObject->GetArrayField("heropc_img");
		if (mirror.Num()>0)
		{
			MirrorChannels.Add(MakeShared<FString>(TEXT("不选择镜像渠道")));
		}
		for (auto& item:mirror)
		{
			auto itemObj = item->AsObject();
			MirrorChannels.Add( MakeShared<FString>(itemObj->GetStringField( "name")) );
		}
	}
	else if (setting->Channel == EHeroUSDKChannel::Global)
	{
		auto mirror = JsonObject->GetArrayField("heroglobalpc_img");
		if (mirror.Num()>0)
		{
			MirrorChannels.Add(MakeShared<FString>(TEXT("不选择镜像渠道")));
		}
		for (auto& item:mirror)
		{
			auto itemObj = item->AsObject();
			MirrorChannels.Add( MakeShared<FString>(itemObj->GetStringField( "name")) );
		}
	}
}

void FHeroUSDKCustomization::CustomizeDetails(IDetailLayoutBuilder& DetailBuilder)
{
	// Create a category so this is displayed early in the properties
	const FString Version = "4.1.0";
	const FString CategoryStr = TEXT("Hero USDK 设置 | 当前版本 ") + Version;
	IDetailCategoryBuilder& ConfigCategory = DetailBuilder.EditCategory(*CategoryStr);
	if(GetMutableDefault<UUsdkSettings>()->UsdkConfig.Len())
	{
		ParseMirror();
	}
	// Add the action buttons
	FDetailWidgetRow& GroupActionsRow = ConfigCategory.AddCustomRow(LOCTEXT("USDK Config Row", "配置行"))
	                                                  .WholeRowContent()
	[
		SNew(SVerticalBox)
		+ SVerticalBox::Slot()
		.HAlign(HAlign_Left)
		.AutoHeight()
		[
			SNew(SHorizontalBox)
			+ SHorizontalBox::Slot()
			  .VAlign(VAlign_Center)
			.AutoWidth()
			[
				SNew(SButton)
					.Text(LOCTEXT("USDK Config Button", "解析配置"))
					.ToolTipText(LOCTEXT("USDK Config Button_Tooltip", "解析中台配置参数"))
					.OnClicked(FOnClicked::CreateStatic(&RequestConfig))
			]
		]

		+ SVerticalBox::Slot()
		.HAlign(HAlign_Left)
		.AutoHeight()
		[
			SNew(SSpacer)
			.Size({10})
			
		]
		
		+ SVerticalBox::Slot()
		.HAlign(HAlign_Left)
		.AutoHeight()
		[
			SNew(SHorizontalBox)
			+SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Center)
			[
				SNew(STextBlock)
				.Text(FText::FromString(TEXT("请选择镜像渠道   ")))
			]
			
			+SHorizontalBox::Slot()
			.AutoWidth()
			.VAlign(VAlign_Center)
			[
				SAssignNew(ComboBox, SComboBox<TSharedPtr<FString>>)
				.OptionsSource(&MirrorChannels)
				.OnGenerateWidget(this, &FHeroUSDKCustomization::OnGenerateComboWidget)
				.OnSelectionChanged(this, &FHeroUSDKCustomization::OnSelectionChanged)
				.InitiallySelectedItem(GetCurrentlySelectedItem())
			   .Content()
			   [
				   SNew(STextBlock)
					 .Text(this, &FHeroUSDKCustomization::GetComboBoxSelectedOptionText)
			   ]
			]

			
			
		]
	];
}


void FHeroUSDKCustomization::GenAndroidCfgFile(const FString& Config)
{
	
	TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject);
	const TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(Config);
	if (!FJsonSerializer::Deserialize(Reader, JsonObject))
	{
		UE_LOG(LogTemp, Error, TEXT("设置安卓配置文件失败，json字符串格式错误,Json:%s"), *Config);
		return;
	}
	
	const FString ProductId = JsonObject->GetStringField(TEXT("productId"));
	const FString ProductKey = JsonObject->GetStringField(TEXT("productKey"));

	if (ProductId.IsEmpty() || ProductKey.IsEmpty())
	{
		UE_LOG(LogTemp, Error, TEXT("设置安卓配置文件失败，ProductId：%s, ProductKey：%s"), *ProductId, *ProductKey);
		return;
	}

	const FString BaseDir = IPluginManager::Get().FindPlugin("HeroUSDKPlugin")->GetBaseDir();
	const FString FilePath = FPaths::ConvertRelativePathToFull(FPaths::Combine(*BaseDir, TEXT("Source/HeroUSDKPlugin/Android/assets/herosdkcfg.xml")));
	if (!FPaths::FileExists(FilePath))
	{
		UE_LOG(LogTemp, Error, TEXT("herosdkcfg.xml 文件不存在,路径：%s"), *FilePath);
		return;
	}

	const TSharedPtr<FXmlFile> File = MakeShareable(new FXmlFile(FilePath));
	if (!File)
	{
		UE_LOG(LogTemp, Error, TEXT("herosdkcfg.xml 文件格式有问题,路径：%s"), *FilePath);
		return;
	}

	const FXmlNode* RootNode = File->GetRootNode();
	TArray<FXmlNode*> Nodes = RootNode->GetChildrenNodes();
	bool bIsEditXml = false;
	for (int32 i = 0; i < Nodes.Num(); i++)
	{
		FXmlNode* NodeItem = Nodes[i];
		FString Key = NodeItem->GetAttribute("name");
		if (Key.Equals("_hu_pid_"))
		{
			NodeItem->SetContent(ProductId);
			bIsEditXml = true;
			UE_LOG(LogTemp, Log, TEXT("------> _hu_pid_ : %s <--------"), *ProductId);
		}
		if (Key.Equals("_hu_pk_"))
		{
			NodeItem->SetContent(ProductKey);
			bIsEditXml = true;
			UE_LOG(LogTemp, Log, TEXT("------> _hu_pk_ : %s <--------"), *ProductKey);
		}
	}
	if (bIsEditXml)
	{
		File->Save(FilePath);
		UE_LOG(LogTemp, Log, TEXT("------> Update herosdkcfg Xml  <--------"));
	}
}


#undef LOCTEXT_NAMESPACE

#endif
