// Copyright Epic Games, Inc. All Rights Reserved.

#include "HeroUSDKPlugin.h"
#include "HeroUsdkSupportKit.h"

#if WITH_EDITOR
#include "Runtime/XmlParser/Public/XmlParser.h"
#include "Runtime/XmlParser/Public/FastXml.h"
#include "ISettingsModule.h"
#endif

#include "HeroUSDKCustomization.h"

#include "LogHeroUsdk.h"
#include "Bili/BiliFunc.h"
#include "Interfaces/IPluginManager.h"
#include "Misc/MessageDialog.h"
#include "Setting/UsdkSettings.h"

#define LOCTEXT_NAMESPACE "FHeroUSDKPluginModule"

#if PLATFORM_WINDOWS

extern BiliSDKInit BiliInitFunc;
extern BiliSDKShowLoginPanel BiliShowLoginPanel;
extern BiliSDKShowPayPanel BiliShowPayPanel;
extern BiliSDKLogout BiliLogout;
extern BiliSDKOpenAntiAddiction BiliOpenAntiAddiction;
extern BiliSDKCloseAccount BiliCloseAccount;
extern BiliSDKUnInit BiliUnInit;

#endif

void FHeroUSDKPluginModule::StartupModule()
{
#if WITH_EDITOR
	FPropertyEditorModule& PropertyModule = FModuleManager::LoadModuleChecked<FPropertyEditorModule>("PropertyEditor");
	PropertyModule.RegisterCustomClassLayout(UUsdkSettings::StaticClass()->GetFName(), FOnGetDetailCustomizationInstance::CreateStatic(&FHeroUSDKCustomization::MakeInstance));
    
	// Register settings
	if (ISettingsModule* SettingsModule = FModuleManager::GetModulePtr<ISettingsModule>("Settings"))
	{
		SettingsModule->RegisterSettings("Project", "Hero", "USDK Config",
			LOCTEXT("Usdk Config", "USDK配置界面"),
			LOCTEXT("Usdk Config", "USDK配置界面"),
										 GetMutableDefault<UUsdkSettings>());
	}
#endif
	FString BaseDir = IPluginManager::Get().FindPlugin("HeroUSDKPlugin")->GetBaseDir();
	FString LibraryPath = FPaths::Combine(*BaseDir, TEXT("Source/HeroUSDKPlugin/Windows/Bili/PCGameSDK.dll"));
#if PLATFORM_WINDOWS
	
	if(auto setting = GetMutableDefault<UUsdkSettings>())
	{
		if(setting->Channel == EHeroUSDKChannel::Bilibili)
		{
			BiliLibraryHandle = !LibraryPath.IsEmpty() ? FPlatformProcess::GetDllHandle(*LibraryPath) : nullptr;

			if (BiliLibraryHandle)
			{
				// Call the test function in the third party library that opens a message box
				BiliInitFunc = (BiliSDKInit)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKInit"));
				BiliShowLoginPanel  = (BiliSDKShowLoginPanel)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKShowLoginPanel"));
				BiliShowPayPanel  = (BiliSDKShowPayPanel)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKShowPayPanel"));
				BiliLogout  = (BiliSDKLogout)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKLogout"));
				BiliOpenAntiAddiction  = (BiliSDKOpenAntiAddiction)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKOpenAntiAddiction"));
				BiliCloseAccount  = (BiliSDKCloseAccount)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKCloseAccount"));
				BiliUnInit  = (BiliSDKUnInit)FPlatformProcess::GetDllExport(BiliLibraryHandle, TEXT("SDKUnInit"));
			}
			else
			{
				FMessageDialog::Open(EAppMsgType::Ok, LOCTEXT("ThirdPartyLibraryError", "Failed to load Bilibili third party library"));
			}
		}
	}
	

#endif
}

void FHeroUSDKPluginModule::ShutdownModule()
{
#if WITH_EDITOR
	if (ISettingsModule* SettingsModule = FModuleManager::GetModulePtr<ISettingsModule>("Settings"))
	{
		SettingsModule->UnregisterSettings("Project", "Hero", "USDK Config");
	}
#endif
#if PLATFORM_WINDOWS
	if(BiliLibraryHandle)
	{
		FPlatformProcess::FreeDllHandle(BiliLibraryHandle);
		BiliLibraryHandle = nullptr;
	}
	
#endif	
}

#undef LOCTEXT_NAMESPACE
	
IMPLEMENT_MODULE(FHeroUSDKPluginModule, HeroUSDKPlugin)

