// Fill out your copyright notice in the Description page of Project Settings.


#include "UWebBrowser.h"

#include "Manager/HeroUsdkSupportPcDelegate.h"
#include "IWebBrowserCookieManager.h"
#include "LogHeroUsdk.h"
#include "WebBrowserModule.h"
#include "SWebBrowser.h"

#define LOCTEXT_NAMESPACE "USDKWebBrowser"


UUWebBrowser::UUWebBrowser(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer), bSupportsTransparency(true)
{
	bIsVariable = true;
}

void UUWebBrowser::LoadURL(FString NewURL)
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->LoadURL(NewURL);
	}
}

void UUWebBrowser::StopLoad()
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->StopLoad();
	}
}

void UUWebBrowser::Reload()
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->Reload();
	}
}

void UUWebBrowser::GoForward()
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->GoForward();
	}
}

void UUWebBrowser::GoBack()
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->GoBack();
	}
}

bool UUWebBrowser::CanGoBack() const
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->CanGoBack();
	}
	return false;
}

bool UUWebBrowser::CanGoForward() const
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->CanGoForward();
	}
	return false;
}

void UUWebBrowser::LoadString(FString Contents, FString DummyURL)
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->LoadString(Contents, DummyURL);
	}
}

void UUWebBrowser::ExecuteJavascript(const FString& ScriptText)
{
	if (WebBrowserWidget.IsValid())
	{
		return WebBrowserWidget->ExecuteJavascript(ScriptText);
	}
}

void UUWebBrowser::BindUObject(const FString & Name, UObject * Object, bool bIsPermanent)
{
	if (WebBrowserWidget.IsValid())
	{
		WebBrowserWidget->BindUObject(Name, Object, bIsPermanent);
	}
}
void UUWebBrowser::OnCleanWebCache()
{
	const IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
	if (WebBrowserSingleton)
	{
		const TSharedPtr<IWebBrowserCookieManager> CookieManager = WebBrowserSingleton->GetCookieManager();
		if (CookieManager.IsValid())
		{
			UE_LOG(LogHeroUsdk, Log, TEXT("UUWebBrowser OnCleanWebCache"));
			CookieManager->DeleteCookies();
		}
	}
}

void UUWebBrowser::DeleteCookies(const FString& URL)
{
	const IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
	if (WebBrowserSingleton)
	{
		const TSharedPtr<IWebBrowserCookieManager> CookieManager = WebBrowserSingleton->GetCookieManager();
		if (CookieManager.IsValid())
		{
			UE_LOG(LogHeroUsdk, Log, TEXT("UUWebBrowser OnCleanWebCache"));
			CookieManager->DeleteCookies(URL);
		}
	}
}


FText UUWebBrowser::GetTitleText() const
{
	if ( WebBrowserWidget.IsValid() )
	{
		return WebBrowserWidget->GetTitleText();
	}

	return FText::GetEmpty();
}

FString UUWebBrowser::GetUrl() const
{
	if (WebBrowserWidget.IsValid())
	{
		return WebBrowserWidget->GetUrl();
	}

	return FString();
}

void UUWebBrowser::ReleaseSlateResources(bool bReleaseChildren)
{
	Super::ReleaseSlateResources(bReleaseChildren);
	if (WebBrowserWidget.IsValid())
	{
		WebBrowserWidget->UnbindInputMethodSystem();
	}
	WebBrowserWidget.Reset();
}

TSharedRef<SWidget> UUWebBrowser::RebuildWidget()
{
	if ( IsDesignTime() )
	{
		return SNew(SBox)
			.HAlign(HAlign_Center)
			.VAlign(VAlign_Center)
			[
				SNew(STextBlock)
				.Text(LOCTEXT("Web Browser", "Web Browser"))
			];
	}
	else
	{
		WebBrowserWidget = SNew(SWebBrowser)
			.InitialURL(InitialURL)
			.ShowErrorMessage(true)
			.ShowControls(false)
			.SupportsTransparency(bSupportsTransparency)
			.OnUrlChanged(BIND_UOBJECT_DELEGATE(FOnTextChanged, HandleOnUrlChanged))
			.OnBeforeNavigation(BIND_UOBJECT_DELEGATE(SWebBrowserView::FOnBeforeBrowse, HandleOnBeforeNavigation))
			.OnBeforePopup(BIND_UOBJECT_DELEGATE(FOnBeforePopupDelegate, HandleOnBeforePopup))
			.OnLoadStarted(BIND_UOBJECT_DELEGATE(FSimpleDelegate,HandleOnLoadStarted))
			.OnLoadCompleted(BIND_UOBJECT_DELEGATE(FSimpleDelegate,HandleOnLoadCompleted))
			.OnLoadError(BIND_UOBJECT_DELEGATE(FSimpleDelegate,HandleOnLoadError))
			.BrowserFrameRate(60);

		// support chinese input
		if (WebBrowserWidget.IsValid())
		{
			class ITextInputMethodSystem* const TextInputMethodSystem = FSlateApplication::Get().GetTextInputMethodSystem();
			WebBrowserWidget->BindInputMethodSystem(TextInputMethodSystem);
		}

		return WebBrowserWidget.ToSharedRef();
	}
}

void UUWebBrowser::SynchronizeProperties()
{
	Super::SynchronizeProperties();

	if ( WebBrowserWidget.IsValid() )
	{

	}
}

void UUWebBrowser::HandleOnLoadCompleted()
{
	UE_LOG(LogHeroUsdk, Log, TEXT("HandleOnLoadCompleted:%s"),*GetUrl());
	OnLoadCompleted.Broadcast();
	
}

void UUWebBrowser::HandleOnLoadStarted()
{
	UE_LOG(LogHeroUsdk, Log, TEXT("HandleOnLoadStarted:%s"),*WebBrowserWidget->GetUrl());
	OnLoadStarted.Broadcast();
}

void UUWebBrowser::HandleOnUrlChanged(const FText& InText)
{
	OnUrlChanged.Broadcast(InText);
	const FString CurrentURL = WebBrowserWidget->GetUrl();
	UE_LOG(LogHeroUsdk, Log, TEXT("HandleOnUrlChanged:%s"),*InText.ToString());
	WebBrowserWidget->ExecuteJavascript(TEXT("window.oncontextmenu = function(event) {event.preventDefault();event.stopPropagation();return false;};"));
}
void UUWebBrowser::HandleOnLoadError()
{
	OnLoadError.Broadcast();
	UE_LOG(LogHeroUsdk, Error, TEXT("UUWebBrowser HandleOnLoadError"));
}

bool UUWebBrowser::HandleOnBeforePopup(FString URL, FString Frame)
{
	UE_LOG(LogHeroUsdk, Log, TEXT("HandleOnBeforePopup,URL:%s,Frame:%s"),*URL,*Frame);
	if (OnBeforePopup.IsBound())
	{
		if (IsInGameThread())
		{
			OnBeforePopup.Broadcast(URL, Frame);
		}
		else
		{
			// Retry on the GameThread.
			TWeakObjectPtr<UUWebBrowser> WeakThis = this;
			FFunctionGraphTask::CreateAndDispatchWhenReady([WeakThis, URL, Frame]()
			{
				if (WeakThis.IsValid())
				{
					WeakThis->HandleOnBeforePopup(URL, Frame);
				}
			}, TStatId(), nullptr, ENamedThreads::GameThread);
		}

		return true;
	}

	return false;
}

bool UUWebBrowser::HandleOnBeforeNavigation(const FString& Url, const FWebNavigationRequest& Request)
{
	UE_LOG(LogHeroUsdk, Log, TEXT("HandleOnBeforeNavigation,Url: %s"), *Url);
	if (OnBeforeNavigation.IsBound())
	{
		if (IsInGameThread())
		{
			OnBeforeNavigation.Broadcast(Url);
		}
		else
		{
			// Retry on the GameThread.
			TWeakObjectPtr<UUWebBrowser> WeakThis = this;
			FFunctionGraphTask::CreateAndDispatchWhenReady([WeakThis, Url, Request]()
			{
				if (WeakThis.IsValid())
				{
					WeakThis->HandleOnBeforeNavigation(Url, Request);
				}
			}, TStatId(), nullptr, ENamedThreads::GameThread);
		}
	}

	return false;
}



#if WITH_EDITOR

const FText UUWebBrowser::GetPaletteCategory()
{
	return LOCTEXT("Experimental", "Experimental");
}

#endif


#undef LOCTEXT_NAMESPACE