// Fill out your copyright notice in the Description page of Project Settings.

#pragma once

#include "CoreMinimal.h"
#include "IWebBrowserWindow.h"
#include "Components/Widget.h"
#include "UWebBrowser.generated.h"

/**
 * 
 */
UCLASS()
class HEROUSDKPLUGIN_API UUWebBrowser : public UWidget
{
	GENERATED_UCLASS_BODY()

	public:
    	DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnUrlChanged, const FText&, Text);
    	DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnBeforePopup, FString, URL, FString, Frame);
		DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnBeforeNavigation, const FString&, Url);
		DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnLoadCompleted);
		DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnLoadStarted);
		DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnLoadError);
	
	/**
	 * Load the specified URL
	 *
	 * @param NewURL New URL to load
	 */
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void LoadURL(FString NewURL);

	/**
	 * Stop load
	 */
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void StopLoad();

	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void Reload();

	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void GoForward();

	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void GoBack();
	
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	bool CanGoBack() const;

	UFUNCTION(BlueprintCallable, Category="Web Browser")
	bool CanGoForward() const;
	/**
	 * Load a string as data to create a web page
	 *
	 * @param Contents String to load
	 * @param DummyURL Dummy URL for the page
	 */
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	void LoadString(FString Contents, FString DummyURL);

	/**
	* Executes a JavaScript string in the context of the web page
	*
	* @param ScriptText JavaScript string to execute
	*/
	UFUNCTION(BlueprintCallable, Category = "Web Browser")
	void ExecuteJavascript(const FString& ScriptText);

	UFUNCTION(BlueprintCallable, Category = "Web Browser")
	void OnCleanWebCache();

	void DeleteCookies(const FString& URL = TEXT(""));
	/**
	 * Get the current title of the web page
	 */
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	FText GetTitleText() const;

	/*
	* Gets the currently loaded URL.
	*
	* @return The URL, or empty string if no document is loaded.
	*/
	UFUNCTION(BlueprintCallable, Category="Web Browser")
	FString GetUrl() const;

	/** Called when the Url changes. */
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnUrlChanged OnUrlChanged;

	/** Called when a popup is about to spawn. */
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnBeforePopup OnBeforePopup;

	/** Called when a popup is about to spawn. */
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnBeforeNavigation OnBeforeNavigation;
	
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnLoadCompleted OnLoadCompleted ;
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnLoadStarted OnLoadStarted;
	UPROPERTY(BlueprintAssignable, Category = "Web Browser|Event")
	FOnLoadError OnLoadError ;
public:

	//~ Begin UWidget interface
	virtual void SynchronizeProperties() override;
	// End UWidget interface

	virtual void ReleaseSlateResources(bool bReleaseChildren) override;

	UFUNCTION(BlueprintCallable, Category = "Web Browser")
	void BindUObject(const FString & Name, UObject * Object, bool bIsPermanent);

	
#if WITH_EDITOR
	virtual const FText GetPaletteCategory() override;
#endif

protected:
	/** URL that the browser will initially navigate to. The URL should include the protocol, eg http:// */
	UPROPERTY(EditAnywhere, Category=Appearance)
	FString InitialURL;

	/** Should the browser window support transparency. */
	UPROPERTY(EditAnywhere, Category=Appearance)
	bool bSupportsTransparency;

protected:
	TSharedPtr<class SWebBrowser> WebBrowserWidget;

protected:
	// UWidget interface
	virtual TSharedRef<SWidget> RebuildWidget() override;
	// End of UWidget interface

	void HandleOnUrlChanged(const FText& Text);
	bool HandleOnBeforePopup(FString URL, FString Frame);
	bool HandleOnBeforeNavigation(const FString& Url, const FWebNavigationRequest& Request);
	void HandleOnLoadCompleted();
	void HandleOnLoadStarted();
	void HandleOnLoadError();



};
