// Copyright (C) 2025 Tencent. All rights reserved.

#ifndef RAIL_SDK_RAIL_DLC_HELPER_H
#define RAIL_SDK_RAIL_DLC_HELPER_H

#include "rail/sdk/rail_dlc_define.h"

namespace rail {
#pragma pack(push, RAIL_SDK_PACKING)

class IRailDlcHelper {
  public:
    // Directly query the specified DLCs permission information owned by the player through the server.
    // If the size of ldc_ids is zero, then all DLCs owned by the player will be returned.
    //
    // Callback is RailQueryIsOwnedDlcsResult.
    virtual RailResult AsyncQueryIsOwnedDlcsOnServer(const RailArray<RailDlcID>& dlc_ids,
                        const RailString& user_data) = 0;

    // Synchronize DLCs status information from the platform. Please note this is the
    // prerequisite for IsDlcInstalled, IsOwnedDlc, GetDlcCount and GetDlcInfo interface.
    //
    // Callback is RailCheckAllDlcsStateReadyResult.
    virtual RailResult AsyncCheckAllDlcsStateReady(const RailString& user_data) = 0;

    // Before calling this interface, you MUST wait for the RailCheckAllDlcsStateReadyResult callback to be received.
    virtual bool IsDlcInstalled(const RailDlcID& dlc_id, RailString* installed_path = NULL) = 0;

    // Before calling this interface, you MUST wait for the RailCheckAllDlcsStateReadyResult callback to be received.
    virtual bool IsOwnedDlc(const RailDlcID& dlc_id) = 0;

    // Before calling this interface, you MUST wait for the RailCheckAllDlcsStateReadyResult callback to be received.
    virtual uint32_t GetDlcCount() = 0;

    // Before calling this interface, you MUST wait for the RailCheckAllDlcsStateReadyResult callback to be received.
    virtual RailResult GetDlcInfo(uint32_t index, RailDlcInfo* dlc_info) = 0;

    // Once the DLC starts installing, you will receive the RailDlcInstallStart callback event.
    // During the DLC installation, you can get the DLC installation progress by receiving
    // the RailDlcInstallProgress callback event.
    // After the DLC is installed, you will receive the RailDlcInstallFinished callback event.
    virtual RailResult AsyncInstallDlc(const RailDlcID& dlc_id, const RailString& user_data) = 0;

    // Callback is RailDlcUninstallFinished.
    virtual RailResult AsyncRemoveDlc(const RailDlcID& dlc_id, const RailString& user_data) = 0;

    // Callback is RailDlcPauseInstall.
    virtual RailResult AsyncPauseInstallDlc(const RailDlcID& dlc_id, const RailString& user_data) = 0;

    // Callback is RailDlcResumeInstall.
    virtual RailResult AsyncResumeInstallDlc(const RailDlcID& dlc_id, const RailString& user_data) = 0;
    
    // Callback is RailDlcInstallFinished and the result parameter will be kErrorDlcCancelInstall.
    virtual RailResult AsyncCancelInstallDlc(const RailDlcID& dlc_id, const RailString& user_data) = 0;

    // Callback is RailShowDlcManageWindowResult.
    virtual RailResult AsyncShowDlcManageWindow(const RailString& user_data) = 0;

    // Callback is RailVerifyDlcIntegrityResult.
    virtual RailResult AsyncVerifyDlcIntegrity(const RailDlcID& dlc_id, const RailString& user_data) = 0;
};

#pragma pack(pop)
}  // namespace rail

#endif  // RAIL_SDK_RAIL_DLC_HELPER_H
