// Copyright (C) 2025 Tencent. All rights reserved.

#ifndef RAIL_SDK_RAIL_PLAYER_DEFINE_H
#define RAIL_SDK_RAIL_PLAYER_DEFINE_H

#include "rail/sdk/rail_event.h"

namespace rail {
#pragma pack(push, RAIL_SDK_PACKING)

enum EnumRailPlayerOnLineState {
    kRailOnlineStateUnknown = 0,
    kRailOnlineStateOffLine = 1,  // player is off-line.
    kRailOnlineStateOnLine  = 2,  // player is on-line.
    kRailOnlineStateBusy    = 3,  // player is on-line, but busy.
    kRailOnlineStateLeave   = 4,  // player is auto away.
    kRailOnlineStateGameDefinePlayingState = 5,  // player is in the game define playing state
};

enum EnumRailPlayerOwnershipType {
    kRailPlayerOwnershipTypeNone = 0,
    kRailPlayerOwnershipTypeOwns = 1,
    kRailPlayerOwnershipTypeFree = 2,
    kRailPlayerOwnershipTypeFreeWeekend = 3,
};

enum EnumRailPlayerBannedStatus {
    kRailPlayerBannedStatusUnknown = 0,
    kRailPlayerBannedStatusNormal = 1,  // player's state is normal
    kRailPlayerBannedStatusBannned = 2,  // player is banned for anti cheat
};

enum RailPlayerAccountType {
    kRailPlayerAccountUnknow = 0,
    kRailPlayerAccountQQ = 1,
    kRailPlayerAccountWeChat = 2,
    kRailPlayerAccountEmail = 3,
    kRailPlayerAccountSteam = 4,
};

struct PlayerPersonalInfo {
    PlayerPersonalInfo() {
        rail_id = 0;
        error_code = kFailure;  // if error_code is not kSuccess, the player's personal
                                // information is not available.
        rail_level = 0;
    }

    RailID rail_id;
    RailResult error_code;
    uint32_t rail_level;
    RailString rail_name;
    RailString avatar_url;
    RailString email_address;
};

struct RailGetAuthenticateURLOptions {
    RailGetAuthenticateURLOptions() {
        client_id = 0;
    }

    RailString url;
    RailString oauth2_state;  // the size of oauth2_state parameter can not more than 64 characters.
    uint64_t client_id;
};

struct RailGetAuthenticateCodeOptions {
    RailGetAuthenticateCodeOptions() {
        client_id = 0;
    }

    uint64_t client_id;
    RailString redirect_uri;
};

// the action field can take several system-defined values, such as:
//    "rail_player_login"
//    "rail_register_character"
//    "rail_start_playing_game"
//
// and you can also set custom action fields for your game.
//
struct RailReportPlayerGameActionOptions {
    RailReportPlayerGameActionOptions() {
        action = "rail_player_login";
        status = "success";
    }

    RailString action;
    RailString status;  // "success" or "failure"
    RailString ext;  // json content type
};

namespace rail_event {

struct AcquireSessionTicketResponse : public RailEvent<kRailEventSessionTicketGetSessionTicket> {
    AcquireSessionTicketResponse() {
        ticket_expire_time = 0;
    }

    RailSessionTicket session_ticket;
    uint32_t ticket_expire_time;  // the session ticket expiration time, in seconds. For example
                                  // return 3600 means expire in one hour. The default validity
                                  // period is 72 hours.
};

struct StartSessionWithPlayerResponse : public RailEvent<kRailEventSessionTicketAuthSessionTicket> {
    StartSessionWithPlayerResponse() {
        remote_rail_id = 0;
    }

    RailID remote_rail_id;
};

// deprecated
struct PlayerGetGamePurchaseKeyResult : public RailEvent<kRailEventPlayerGetGamePurchaseKey> {
    PlayerGetGamePurchaseKeyResult() {
        result = kErrorPlayerGameNotSupportPurchaseKey;
    }

    RailString purchase_key;
};

struct QueryPlayerBannedStatus : public RailEvent<kRailEventQueryPlayerBannedStatus> {
    QueryPlayerBannedStatus() {
        status = kRailPlayerBannedStatusUnknown;
    }

    EnumRailPlayerBannedStatus status;
};

struct GetAuthenticateURLResult : public RailEvent<kRailEventPlayerGetAuthenticateURL> {
    GetAuthenticateURLResult() {
        result = kFailure;
        ticket_expire_time = 0;
    }

    RailString source_url;
    RailString authenticate_url;
    uint32_t ticket_expire_time;  // UTC time
};

// deprecated
struct RailAntiAddictionGameOnlineTimeChanged
    : public RailEvent<kRailEventPlayerAntiAddictionGameOnlineTimeChanged> {
    RailAntiAddictionGameOnlineTimeChanged() {
        game_online_time_count_minutes = 0;
    }

    uint32_t game_online_time_count_minutes;
};

struct RailGetEncryptedGameTicketResult
    : public RailEvent<kRailEventPlayerGetEncryptedGameTicketResult> {
    RailGetEncryptedGameTicketResult() {
        result = kFailure;
    }

    RailString encrypted_game_ticket;
};

struct RailGetPlayerMetadataResult
    : public RailEvent<kRailEventPlayerGetPlayerMetadataResult> {
    RailGetPlayerMetadataResult() {
        result = kFailure;
    }

    RailArray<RailKeyValue> key_values;
};

struct RailGetAuthenticateCodeResult
    : public RailEvent<kRailEventPlayerGetAuthenticateCodeResult> {
    RailGetAuthenticateCodeResult() {
        code_expire_time = 0;
    }

    RailString code;
    uint32_t code_expire_time;  // UTC time
};

struct RailQueryPlayerCollegeCertificateResult
    : public RailEvent<kRailEventPlayerQueryPlayerCollegeCertificateResult> {
    RailQueryPlayerCollegeCertificateResult() {
        college_student = false;
        certificate_state = 0;
    }

    bool college_student;
    uint32_t certificate_state;
    RailString college_name;
};

}  // namespace rail_event

#pragma pack(pop)
}  // namespace rail

#endif  // RAIL_SDK_RAIL_PLAYER_DEFINE_H
