// Copyright (C) 2025 Tencent. All rights reserved.

#ifndef RAIL_SDK_RAIL_VOICE_CHANNEL_H
#define RAIL_SDK_RAIL_VOICE_CHANNEL_H

#include "rail/sdk/base/rail_component.h"
#include "rail/sdk/rail_voice_channel_define.h"

// @desc To use the IRailVoiceChannel APIs, you will first enable the feature by contacting the
// platform. In future, enabling the feature might be configurable on the developer portal.

namespace rail {
#pragma pack(push, RAIL_SDK_PACKING)

class IRailVoiceChannel;
class IRailVoiceHelper {
  public:
    virtual ~IRailVoiceHelper() {}

    // create an empty channel and won't join it
    // trigger event CreateVoiceChannelResult
    // @param options creation options
    // @param channel_name used to show the readable name when inviting others
    //        maintained by game itself
    // @param result return the request result, the final creation result will be sent by callback
    // @return IRailVoiceChannel NULL if failed
    virtual IRailVoiceChannel* AsyncCreateVoiceChannel(const CreateVoiceChannelOption& options,
                                const RailString& channel_name,
                                const RailString& user_data,
                                RailResult* result) = 0;

    // open an existing voice channel, but won't join it
    // @param voice_channel_id an existing voice channel id
    // @param channel_name used to show the readable name when inviting others
    //        maintained by game itself
    // @param result return the open result
    // @return IRailVoiceChannel NULL if failed, it will still return the IRailVoiceChannel even if
    // the voice_channel_id is not exist, in this case, all function of IRailVoiceChannel will fail
    virtual IRailVoiceChannel* OpenVoiceChannel(const RailVoiceChannelID& voice_channel_id,
                                const RailString& channel_name,
                                RailResult* result) = 0;

    // return the current state of speaker
    virtual EnumRailVoiceChannelSpeakerState GetSpeakerState() = 0;

    // request to mute the speaker
    virtual RailResult MuteSpeaker() = 0;

    // request to resume the speaker
    virtual RailResult ResumeSpeaker() = 0;

    // deprecated
    virtual RailResult SetupVoiceCapture(const RailVoiceCaptureOption& options,
                        RailCaptureVoiceCallback callback = NULL) = 0;

    // deprecated
    virtual RailResult StartVoiceCapturing(uint32_t duration_milliseconds = 120,
                        bool repeat = true) = 0;

    // deprecated
    virtual RailResult StopVoiceCapturing() = 0;

    // deprecated
    virtual RailResult GetCapturedVoiceData(void* buffer,
                        uint32_t buffer_length,
                        uint32_t* encoded_bytes_written) = 0;

    // deprecated
    virtual RailResult DecodeVoice(const void* encoded_buffer,
                        uint32_t encoded_length,
                        void* pcm_buffer,
                        uint32_t pcm_buffer_length,
                        uint32_t* pcm_buffer_written) = 0;

    // deprecated
    virtual RailResult GetVoiceCaptureSpecification(RailVoiceCaptureSpecification* spec) = 0;

    // deprecated
    virtual RailResult EnableInGameVoiceSpeaking(bool can_speaking) = 0;

    // set the nickname used to show on the overlay UI
    // if you don't call it, the name will be the platform nickname
    // so you must call this if players have a different in-game nickname
    virtual RailResult SetPlayerNicknameInVoiceChannel(const RailString& nickname) = 0;

    // set the hot key of microphone, player could press hot key to speak in channel
    // if you don't call it, there will be a default hot key could be seen on overlay UI
    // @param push_to_talk_hot_key Virtual-Key Code on windows
    // @see https://msdn.microsoft.com/en-us/library/windows/desktop/dd375731(v=vs.85).aspx
    virtual RailResult SetPushToTalkKeyInVoiceChannel(uint32_t push_to_talk_hot_key) = 0;

    // get the hot key of microphone
    // @return Virtual-Key Code on windows, zero if invalid
    virtual uint32_t GetPushToTalkKeyInVoiceChannel() = 0;

    // show the overlay ui provided by WeGame, players could do simple operations of voice features
    // it was not shown by default
    virtual RailResult ShowOverlayUI(bool show) = 0;

    virtual RailResult SetMicroVolume(uint32_t volume) = 0;

    virtual RailResult SetSpeakerVolume(uint32_t volume) = 0;
};

class IRailVoiceChannel : public IRailComponent {
  public:
    virtual ~IRailVoiceChannel() {}

    // get current channel's id, if this channel hasn't been created, it will return an invalid id
    virtual RailVoiceChannelID GetVoiceChannelID() = 0;

    // the name is same to the parameter passed by AsyncCreateVoiceChannel or OpenVoiceChannel
    virtual const RailString& GetVoiceChannelName() = 0;

    // if you don't want to listen events to get the result of async functions' callbacks
    // you could check this to get the real state of current player
    virtual EnumRailVoiceChannelJoinState GetJoinState() = 0;

    // request to join channel, will trigger callback JoinVoiceChannelResult to get the join result
    virtual RailResult AsyncJoinVoiceChannel(const RailString& user_data) = 0;

    // request to leave this channel, will trigger callback LeaveVoiceChannelResult with reason
    // kRailVoiceLeaveChannelReasonExitedBySelf
    virtual RailResult AsyncLeaveVoiceChannel(const RailString& user_data) = 0;

    // get current users in this channel
    // you could listen event VoiceChannelMemeberChangedEvent to get users list more effective
    virtual RailResult GetUsers(RailArray<RailID>* user_list) = 0;

    // request to invite other users to join in this channel
    // callback: VoiceChannelAddUsersResult
    // the other users would reject the invitation even callback return success
    // player be invited will receive the VoiceChannelInviteEvent and need join channel manually
    virtual RailResult AsyncAddUsers(const RailArray<RailID>& user_list,
                        const RailString& user_data) = 0;

    // deprecated
    virtual RailResult AsyncRemoveUsers(const RailArray<RailID>& user_list,
                        const RailString& user_data) = 0;

    // deprecated
    virtual RailResult CloseChannel() = 0;

    // set whether current player could speak to others
    // pass true to this api just like pressing the hot key to allow player to speak,
    // and pass false to this api just like releasing the hot key to finish the speaking
    virtual RailResult SetSelfSpeaking(bool speaking) = 0;

    // get whether current player is speaking
    virtual bool IsSelfSpeaking() = 0;

    // set the users' speaking state to you, you could set users' speaking state incrementally
    // retrieve callback to get the real result
    // callback: AsyncSetUsersSpeakingState
    virtual RailResult AsyncSetUsersSpeakingState(
                        const RailArray<RailVoiceChannelUserSpeakingState>& users_speaking_state,
                        const RailString& user_data) = 0;

    // get all users' speaking state in this channel
    virtual RailResult GetUsersSpeakingState(
                        RailArray<RailVoiceChannelUserSpeakingState>* users_speaking_state) = 0;

    // get users currently speaking, you could listen the event
    // VoiceChannelSpeakingUsersChangedEvent to avoid call this api periodically
    // @speaking_users speaking users
    // @others convenient way to get the users not speaking
    virtual RailResult GetSpeakingUsers(RailArray<RailID>* speaking_users,
                        RailArray<RailID>* not_speaking_users) = 0;

    // deprecated
    virtual bool IsOwner() = 0;
};

#pragma pack(pop)
}  // namespace rail

#endif  // RAIL_SDK_RAIL_VOICE_CHANNEL_H
